﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Runtime.InteropServices;
using System.Threading;

namespace UHFDemo
{
    public partial class Form_Main : Form
    {
        /* Tag data struct type */
        [StructLayout(LayoutKind.Sequential)]
        public struct TagDataStructType {
	        public UInt32	tagIndex;					/* record for the row of tag table */
	        public byte	    rssi;						/* receive rssi */
	        public UInt32	freq;						/* receive frequency */
	        public UInt16	reflectPower;				/* reflected power */
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = 256)]
	        public byte[]	data;		                /* tag data area */
	        public byte	    dataLen;					/* tag data length */
        };

        public delegate void CallbackDelegate(ref TagDataStructType tag);
        [DllImport("UhfDll.dll", CallingConvention = CallingConvention.Cdecl)]
        public static extern bool AS3992_Connect(UInt32 port, ulong baudrate);
        [DllImport("UhfDll.dll", CallingConvention = CallingConvention.Cdecl)]
        public static extern bool AS3992_Disconnect();
        [DllImport("UhfDll.dll", CallingConvention = CallingConvention.Cdecl)]
        public static extern bool AS3992_CallAntennaPower();
        [DllImport("UhfDll.dll", CallingConvention = CallingConvention.Cdecl)]
        public static extern void AS3992_CallInventoryRSSI_Single(CallbackDelegate call);
        [DllImport("UhfDll.dll", CallingConvention = CallingConvention.Cdecl)]
        public static extern void AS3992_CallInventoryRSSI_Stop(CallbackDelegate call);
        [DllImport("UhfDll.dll", CallingConvention = CallingConvention.Cdecl)]
        public static extern bool AS3992_CallSelectTag(byte[] epc, byte epclen);
        [DllImport("UhfDll.dll", CallingConvention = CallingConvention.Cdecl)]
        public static extern bool AS3992_CallConfigRFPower(byte attenuation, ref byte cur_attenuation);
        [DllImport("UhfDll.dll", CallingConvention = CallingConvention.Cdecl)]
        public static extern bool AS3992_CallSetSensitivity(byte sensitivity, ref byte cur_sensitivity);

        struct TagRecord
        {
            public int tagIndex;
            public string tagId;
            public string tagRSSI;
            public int tagCount;
        }

        private UInt32[] baudrates = { 9600, 14400, 19200, 38400, 56000, 57600, 115200, 128000, 256000 };
        private LabelTrackBar ltbPower, ltbSensitivity;
        private int curTagIndex;
        private List<TagRecord> listTagRecord;
        private bool isScan = false;

        private delegate void RefreshDelegate();
        private RefreshDelegate refreshDelegate;

        public Form_Main()
        {
            InitializeComponent();

            string[] portNames = System.IO.Ports.SerialPort.GetPortNames();
            foreach (string name in portNames) {
                this.comboBox_port.Items.Add(name);
            }
            if (portNames.Length > 0)
            {
                this.comboBox_port.SelectedIndex = 0;
            }

            string[] mBaudrates = { "9600", "14400", "19200", "38400", "56000", "57600", "115200", "128000", "256000" };
            foreach (string baudrate in mBaudrates)
            {
                this.comboBox_baudrate.Items.Add(baudrate);
            }
            this.comboBox_baudrate.SelectedIndex = 6;

            this.listView_tag.FullRowSelect = true;
            this.listView_tag.View = View.Details;
            this.listView_tag.HeaderStyle = ColumnHeaderStyle.Nonclickable;
            this.listView_tag.Columns.Add("NO", 35, HorizontalAlignment.Center);
            this.listView_tag.Columns.Add("Tag ID", 240, HorizontalAlignment.Center);
            this.listView_tag.Columns.Add("RSSI", 80, HorizontalAlignment.Center);
            this.listView_tag.Columns.Add("Reflected Power", 95, HorizontalAlignment.Center);
            this.listView_tag.Columns.Add("Count", 60, HorizontalAlignment.Center);
            this.listView_tag.Visible = true;

            ltbPower = new LabelTrackBar(0);
            ltbPower.OnTractBar_MouseUp += new LabelTrackBar.ScrollHandler(ltbPower_OnTractBar_MouseUp);
            this.panel_power.Controls.Add(ltbPower);

            ltbSensitivity = new LabelTrackBar(1);
            ltbSensitivity.OnTractBar_MouseUp += new LabelTrackBar.ScrollHandler(ltbSensitivity_OnTractBar_MouseUp);
            this.panel_sensitivity.Controls.Add(ltbSensitivity);

            refreshDelegate = new RefreshDelegate(refreshListView);

            this.button_single.Enabled = false;
            this.button_continuous.Enabled = false;
            this.button_readtag.Enabled = false;
        }

        private void ltbPower_OnTractBar_MouseUp(object sender)
        {
            byte attenuation = 0;
            AS3992_CallConfigRFPower((byte)ltbPower.TickValue, ref attenuation);
        }

        private void ltbSensitivity_OnTractBar_MouseUp(object sender)
        {
            byte sensitivity = 0;
            AS3992_CallConfigRFPower((byte)ltbSensitivity.TickValue, ref sensitivity);
        }

        private void CallbackFunc(ref TagDataStructType tag)
        {
            int i;
            string strTagId = "";

            for (i = 0; i < tag.dataLen - 1; i++)
            {
                strTagId += String.Format("{0:X2}-", tag.data[i]);
            }
            strTagId += String.Format("{0:X2}", tag.data[i]);
            for (i = 0; i < listTagRecord.Count; i++)
            {
                if (listTagRecord[i].tagId.Contains(strTagId))
                {
                    TagRecord TagRecord = listTagRecord[i];
                    TagRecord.tagCount++;
                    listTagRecord[i] = TagRecord;
                    break;
                }
            }
            if (i == listTagRecord.Count)
            {
                TagRecord tagRecord = new TagRecord();
                tagRecord.tagIndex = ++curTagIndex;
                tagRecord.tagId = strTagId;
                tagRecord.tagRSSI = String.Format("{0:P}", (tag.rssi / 255.0));
                tagRecord.tagCount = 1;
                listTagRecord.Add(tagRecord);
            }
            refreshDelegate();
        }

        private void refreshListView()
        {
            if (this.listView_tag.InvokeRequired)
            {
                this.Invoke(refreshDelegate, null);
            }
            else
            {
                this.listView_tag.BeginUpdate();
                this.listView_tag.Items.Clear();
                for (int i = 0; i < listTagRecord.Count; i++)
                {
                    ListViewItem lvItem = new ListViewItem();
                    lvItem.Text = listTagRecord[i].tagIndex.ToString();
                    lvItem.SubItems.Add(listTagRecord[i].tagId);
                    lvItem.SubItems.Add(listTagRecord[i].tagRSSI);
                    lvItem.SubItems.Add("00.00%");
                    lvItem.SubItems.Add(listTagRecord[i].tagCount.ToString());
                    this.listView_tag.Items.Add(lvItem);
                }
                this.listView_tag.EndUpdate();
                this.listView_tag.Refresh();
            }
        }

        public static byte[] HexStringtoHex(string hexString)
        {
            int len = hexString.Length / 2;
            byte[] hex = new byte[len];
            for (int i = 0; i < len; i++)
            {
                hex[i] = Convert.ToByte(hexString.Substring(i * 2, 2), 16);
            }
            return hex;
        }

        private void ContinuousScan()
        {
            while (isScan)
            {
                AS3992_CallInventoryRSSI_Single(CallbackFunc);
                System.Threading.Thread.Sleep(100);
            }
        }

        private void button_connect_Click(object sender, EventArgs e)
        {
            byte port = (byte)(int.Parse(this.comboBox_port.Text.Remove(0, 3)));
            if (this.button_connect.Text == "Connect")
            {
                if (AS3992_Connect(port, baudrates[this.comboBox_baudrate.SelectedIndex]))
                {
                    byte cur_attenuation = 0, cur_sensitivity = 0;
                    if (AS3992_CallConfigRFPower(0xFF, ref cur_attenuation) && AS3992_CallSetSensitivity(0xFF, ref cur_sensitivity))
                    {
                        this.button_connect.Text = "DisConnect";
                        this.button_single.Enabled = true;
                        this.button_continuous.Enabled = true;
                        this.button_readtag.Enabled = true;
                        ltbPower.TickValue = cur_attenuation;
                        ltbSensitivity.TickValue = cur_sensitivity;
                    }
                    else
                    {
                        AS3992_Disconnect();
                    }
                }
            }
            else
            {
                if (AS3992_Disconnect())
                {
                    this.button_connect.Text = "Connect";
                    this.button_single.Enabled = false;
                    this.button_continuous.Enabled = false;
                    this.button_readtag.Enabled = false;
                }
            }
        }

        private void button2_Click(object sender, EventArgs e)
        {
            listTagRecord = new List<TagRecord>();
            curTagIndex = 0;
            AS3992_CallInventoryRSSI_Single(CallbackFunc);
        }

        private void button3_Click(object sender, EventArgs e)
        {
            if (this.button_continuous.Text == "Continuous")
            {
                listTagRecord = new List<TagRecord>();
                curTagIndex = 0;
                this.button_continuous.Text = "StopScan";
                isScan = true;
                Thread ScanThread = new Thread(new ThreadStart(ContinuousScan));
                ScanThread.IsBackground = true;
                ScanThread.Start();
            }
            else
            {
                isScan = false;
                AS3992_CallInventoryRSSI_Stop(CallbackFunc);
                this.button_continuous.Text = "Continuous";
                System.Threading.Thread.Sleep(1000);
            }
        }

        private void button_readtag_Click(object sender, EventArgs e)
        {
            if (this.listView_tag.FocusedItem != null)
            {
                if (this.listView_tag.SelectedItems.Count == 1)
                {
                    string strEPC = listTagRecord[this.listView_tag.SelectedItems[0].Index].tagId;
                    strEPC = strEPC.Replace("-", "");
                    byte[] EPC = HexStringtoHex(strEPC);
                    //MessageBox.Show("" + EPC.Length);
                    AS3992_CallSelectTag(EPC, (byte)EPC.Length);

                    RdWrDialog dialog = new RdWrDialog();
                    dialog.StartPosition = FormStartPosition.CenterParent;
                    dialog.ShowDialog(this);
                    return;
                }
            }
            MessageBox.Show("Please Select one tag before !");
        }

        private void button4_Click(object sender, EventArgs e)
        {
            this.listView_tag.Items.Clear();
        }

    }
}
